/*
* ms_complex_clk.c- Sigmastar
*
* Copyright (C) 2018 Sigmastar Technology Corp.
*
* Author: karl.xiao <karl.xiao@sigmastar.com.tw>
*
* This software is licensed under the terms of the GNU General Public
* License version 2, as published by the Free Software Foundation, and
* may be copied, distributed, and modified under those terms.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
*/
#include <linux/kernel.h>
#include <linux/clkdev.h>
#include <linux/clk-provider.h>
#include <linux/clk.h>
#include <linux/of_address.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/cpu.h>
#include <linux/delay.h>

#include "ms_types.h"
#include "ms_platform.h"
#include "infinity2m/registers.h"

#if defined (CONFIG_MS_CPU_FREQ) && defined (CONFIG_MS_GPIO)
extern u8 enable_scaling_voltage;
#include "infinity2m/gpio.h"
#include "../../gpio/mdrv_gpio.h"
int vid_0 = -1;
int vid_1 = -1;
#else
u8 enable_scaling_voltage=0;
#define MDrv_GPIO_Set_High(x) {}
#define MDrv_GPIO_Set_Low(x) {}
#define MDrv_GPIO_Pad_Set(x) {}
#define MDrv_GPIO_Pad_Odn(x) {}
int vid_0 = -1;
int vid_1 = -1;
#endif

#define CLK_DEBUG  0

#if CLK_DEBUG
#define CLK_DBG(fmt, arg...) printk(KERN_DEBUG fmt, ##arg)
#else
#define CLK_DBG(fmt, arg...)
#endif
#define CLK_ERR(fmt, arg...) printk(KERN_ERR fmt, ##arg)

static unsigned int ms_get_ddr_scl(void)
{
    unsigned int factor = INREG16(BASE_REG_RIU_PA + (0x1032A0 << 1)) | (((INREG16(BASE_REG_RIU_PA + (0x1032A2 << 1)) & 0xFF) <<16));

    if(!factor)
        factor = (INREG16(0x1F206580) | ((INREG16(0x1F206584) & 0xFF) << 16));

    CLK_DBG("ms_get_ddr_scl = 0x%X\n", factor);
    return factor;
}

static long ms_cpuclk_round_rate(struct clk_hw *clk_hw, unsigned long rate, unsigned long *parent_rate)
{
    //CLK_DBG("ms_cpuclk_round_rate = %lu\n", rate);

    if(rate <= 450000000)       //request 400M-450M=400M
    {
        return 400000000;
    }
    else if(rate <= 650000000)  //request 450M-650M=600M
    {
        return 600000000;
    }
    else if(rate <= 850000000)  //request 650M-850M=800M
    {
        return 800000000;
    }
    else if(rate <= 1000000000) //request 800M-1000M=1000M
    {
        return 1000000000;
    }
    else                        //request 1000M-1200M=1200M
    {
        return 1200000000;
    }
}

static unsigned long ms_cpuclk_recalc_rate(struct clk_hw *clk_hw, unsigned long parent_rate)
{
    unsigned long rate;

    rate = (parent_rate / ms_get_ddr_scl()) << 23;

    CLK_DBG("ms_cpuclk_recalc_rate = %lu, prate=%lu\n", rate, parent_rate);

    return rate;
}

void cpu_dvfs(U32 u32TargetSet)
{
#if 1
    OUTREG8(BASE_REG_RIU_PA + (0x1032C2 << 1), (u32TargetSet>>16)&0xFF);
    OUTREG8(BASE_REG_RIU_PA + (0x1032C1 << 1), (u32TargetSet>>8)&0xFF);
    OUTREG8(BASE_REG_RIU_PA + (0x1032C0 << 1), (u32TargetSet)&0xFF);
    OUTREG8(BASE_REG_RIU_PA + (0x1032C4 << 1), 0x01);//Update ARM frequency synthesizer N.f setting
    OUTREG8(BASE_REG_RIU_PA + (0x103223 << 1), 0x00);//ARM PLL PD (Power-on)
#else
    OUTREG16(BASE_REG_RIU_PA + (0x1032A4 << 1), u32TargetSet&0xFFFF); //set target freq to LPF high
    OUTREG16(BASE_REG_RIU_PA + (0x1032A6 << 1), (u32TargetSet>>16)&0xFFFF);
    OUTREG16(BASE_REG_RIU_PA + (0x1032B0 << 1), 0x0001); //switch to LPF control
    OUTREG16(BASE_REG_RIU_PA + (0x1032AA << 1), 0x0006); //mu[2:0]
    OUTREG16(BASE_REG_RIU_PA + (0x1032AE << 1), 0x0008); //lpf_update_cnt[7:0]
    SETREG16(BASE_REG_RIU_PA + (0x1032B2 << 1), BIT12);  //from low to high
    OUTREG16(BASE_REG_RIU_PA + (0x1032A8 << 1), 0x0000); //toggle LPF enable
    OUTREG16(BASE_REG_RIU_PA + (0x1032A8 << 1), 0x0001);
    while( !(INREG16(BASE_REG_RIU_PA + (0x1032BA << 1))&BIT0) ); //polling done
    OUTREG16(BASE_REG_RIU_PA + (0x1032A8 << 1), 0x0000);
    OUTREG16(BASE_REG_RIU_PA + (0x1032A0 << 1), u32TargetSet&0xFFFF);  //store freq to LPF low
    OUTREG16(BASE_REG_RIU_PA + (0x1032A2 << 1), (u32TargetSet>>16)&0xFFFF);
#endif
}



#define VOLTAGE_CORE_850   850
#define VOLTAGE_CORE_900   900
#define VOLTAGE_CORE_950   950
#define VOLTAGE_CORE_1000 1000
#define VOLTAGE_CORE_NO_CTRL 0xDEAD

int g_sCurrentTemp = 35;
int g_sCurrentTempThreshLo=40;  // T < 40C : VDD = 1.0V
int g_sCurrentTempThreshHi=60;  // T > 60C : VDD= 0.9V
int g_sCurrentVoltageCore=VOLTAGE_CORE_1000;
void set_core_voltage_nvr (int vcore);
void set_core_voltage_p2 (int vcore);
void (*set_core_voltage) (int vcore) = set_core_voltage_nvr;

/* NVR Vcore control
 * ===========================
 * | VID_0 | VID_1 | 1V0_STD |
 * ---------------------------
 * |   L   |   L   | 0.855v  |
 * |   H   |   L   | 0.903v  |
 * |   L   |   H   | 0.955v  |
 * |   H   |   H   | 1.005v  |
 * |   IN  |   IN  | NO_CTRL |
 * ===========================
 */
void set_core_voltage_nvr (int vcore)
{
    if(enable_scaling_voltage)
    {
        if(g_sCurrentVoltageCore == vcore) {
            return;
        }

        switch (vcore)
        {
        case VOLTAGE_CORE_850:
            if(vid_0 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_0);
                MDrv_GPIO_Set_Low(vid_0);
            }
            if(vid_1 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_1);
                MDrv_GPIO_Set_Low(vid_1);
            }
            g_sCurrentVoltageCore = VOLTAGE_CORE_850;
            break;

        case VOLTAGE_CORE_900:
            if(vid_0 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_0);
                MDrv_GPIO_Set_High(vid_0);
            }
            if(vid_1 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_1);
                MDrv_GPIO_Set_Low(vid_1);
            }
            g_sCurrentVoltageCore = VOLTAGE_CORE_900;
            break;

        case VOLTAGE_CORE_950:
            if(vid_0 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_0);
                MDrv_GPIO_Set_Low(vid_0);
            }
            if(vid_1 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_1);
                MDrv_GPIO_Set_High(vid_1);
            }
            g_sCurrentVoltageCore = VOLTAGE_CORE_950;
            break;

        case VOLTAGE_CORE_1000:
            if(vid_0 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_0);
                MDrv_GPIO_Set_High(vid_0);
            }
            if(vid_1 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_1);
                MDrv_GPIO_Set_High(vid_1);
            }
            g_sCurrentVoltageCore = VOLTAGE_CORE_1000;
            break;

        case VOLTAGE_CORE_NO_CTRL:
            // FIXME: temporary not to control Vcore
            printk("%s: >>>> set vid_0/vid_1 as GPIO/INPUT !!!\n", __func__);
            if(vid_0 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_0);
                MDrv_GPIO_Pad_Odn(vid_0);
            }
            if(vid_1 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_1);
                MDrv_GPIO_Pad_Odn(vid_1);
            }
            printk("%s: <<<< set vid_0/vid_1 as GPIO/INPUT !!!\n", __func__);
            g_sCurrentVoltageCore = VOLTAGE_CORE_1000;
            break;

        }
        CLK_DBG("CurrentVoltageCore = %d\n", g_sCurrentVoltageCore);
    }
}

/* P2 Vcore control
 * ===========================
 * | VID_1 | VID_0 | 1V0_STD |
 * ---------------------------
 * |   L   |   L   | 0.855v  |
 * |   H   |   L   | 0.903v  |
 * |   L   |   H   | 0.955v  |
 * |   H   |   H   | 1.005v  |
 * |   IN  |   IN  | NO_CTRL |
 * ===========================
 */
void set_core_voltage_p2 (int vcore)
{
    if(enable_scaling_voltage)
    {
        if(g_sCurrentVoltageCore == vcore) {
            return;
        }

        switch (vcore)
        {
        case VOLTAGE_CORE_850:
            if(vid_0 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_0);
                MDrv_GPIO_Set_Low(vid_0);
            }
            if(vid_1 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_1);
                MDrv_GPIO_Set_Low(vid_1);
            }
            g_sCurrentVoltageCore = VOLTAGE_CORE_850;
            break;

        case VOLTAGE_CORE_900:
            if(vid_0 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_0);
                MDrv_GPIO_Set_Low(vid_0);
            }
            if(vid_1 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_1);
                MDrv_GPIO_Set_High(vid_1);
            }
            g_sCurrentVoltageCore = VOLTAGE_CORE_900;
            break;

        case VOLTAGE_CORE_950:
            if(vid_0 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_0);
                MDrv_GPIO_Set_High(vid_0);
            }
            if(vid_1 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_1);
                MDrv_GPIO_Set_Low(vid_1);
            }
            g_sCurrentVoltageCore = VOLTAGE_CORE_950;
            break;

        case VOLTAGE_CORE_1000:
            if(vid_0 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_0);
                MDrv_GPIO_Set_High(vid_0);
            }
            if(vid_1 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_1);
                MDrv_GPIO_Set_High(vid_1);
            }
            g_sCurrentVoltageCore = VOLTAGE_CORE_1000;
            break;
        case VOLTAGE_CORE_NO_CTRL:
            // FIXME: temporary not to control Vcore
            printk("%s: >>>> set vid_0/vid_1 as GPIO/INPUT !!!\n", __func__);
            if(vid_0 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_0);
                MDrv_GPIO_Pad_Odn(vid_0);
            }
            if(vid_1 != -1)
            {
                MDrv_GPIO_Pad_Set(vid_1);
                MDrv_GPIO_Pad_Odn(vid_1);
            }
            printk("%s: <<<< set vid_0/vid_1 as GPIO/INPUT !!!\n", __func__);
            g_sCurrentVoltageCore = VOLTAGE_CORE_1000;
            break;

        }
        CLK_DBG("CurrentVoltageCore = %d\n", g_sCurrentVoltageCore);
    }
}

static int ms_cpuclk_set_rate(struct clk_hw *clk_hw, unsigned long rate, unsigned long parent_rate)
{
    int ret = 0;

    //if(abs(rate - clk_get_rate(clk_hw->clk)) < 100000000)  //if it's the same setting, just return to save time
    //    return 0;


    printk("%s: rate = %lu\n",__func__, rate);

    if(rate == 1000000000)
    {
        set_core_voltage(VOLTAGE_CORE_900);
        cpu_dvfs(0x374BC7);
    }
    else if(rate == 1200000000)
    {
        set_core_voltage(VOLTAGE_CORE_1000);
        cpu_dvfs(0x2E147B);
    }
    else
    {
        CLK_DBG("cur_temp = %d\n", g_sCurrentTemp);
        #if 1
        if((g_sCurrentVoltageCore==VOLTAGE_CORE_1000) && (g_sCurrentTemp>g_sCurrentTempThreshHi))
            set_core_voltage(VOLTAGE_CORE_950);
        if((g_sCurrentVoltageCore==VOLTAGE_CORE_950) && (g_sCurrentTemp<g_sCurrentTempThreshLo))
            set_core_voltage(VOLTAGE_CORE_1000);
        #else
        if(g_sCurrentVoltageCore==VOLTAGE_CORE_1000)
            set_core_voltage(VOLTAGE_CORE_950);
        #endif

        if(rate == 400000000)
        {
            cpu_dvfs(0x8A3D71);
        }
        else if(rate == 600000000)
        {
            cpu_dvfs(0x5C28F6);
        }
        else if(rate == 800000000)
        {
            cpu_dvfs(0x451EB8);
        }
    }

    return ret;
}

#define BOND_SSI210_NVR_512Mb_DDR2_ESMT 0x1D
#define BOND_SSI210_NVR_512Mb_DDR2_Nanya 0x19
#define BOND_SSI220_P2_512Mb_DDR2_ESMT 0x1D
#define BOND_SSI220_P2_512Mb_DDR2_Nanya 0x19
#define BOND_SSI220H_P2_256Mb_DDR2_Winbond 0x1C
#define BOND_SSI220H_P2_256Mb_DDR2_ESMT 0x18
#define BOND_SSI210D_NVR_1024Mb_DDR3_Nanya 0x1E
#define BOND_SSI210D_NVR_1024Mb_DDR3_Winbond 0x1A
#define BOND_SSI220D_P2_1024Mb_DDR3_Nanya 0x1E
#define BOND_SSI220D_P2_1024Mb_DDR3_Winbond 0x1A
#define BOND_SSI210Q_NVR_2048Mb_DDR3_Nanya 0x1F
#define BOND_SSI210Q_NVR_2048Mb_DDR3_Winbond 0x1B
#define BOND_SSI215_NVR_512Mb_DDR2_ESMT 0x15
#define BOND_SSI230G_NVR_EXT_DDR 0x00

static void ms_vid_ctrl_fn_init(void)
{
    unsigned short bonding = INREG16(BASE_REG_RIU_PA + (0x101E90 << 1));
    switch(bonding)
    {
        case BOND_SSI210D_NVR_1024Mb_DDR3_Nanya:
        case BOND_SSI210D_NVR_1024Mb_DDR3_Winbond:
        case BOND_SSI210Q_NVR_2048Mb_DDR3_Nanya:
        case BOND_SSI210Q_NVR_2048Mb_DDR3_Winbond:
            printk("%s bonding 0x%x=================> NVR\n", __func__, bonding );
            set_core_voltage = set_core_voltage_nvr;
            break;
        case BOND_SSI220_P2_512Mb_DDR2_ESMT:
        case BOND_SSI220_P2_512Mb_DDR2_Nanya:
            printk("%s bonding 0x%x=================> P2\n", __func__, bonding );
            set_core_voltage = set_core_voltage_p2;
            break;
    }
}

void ms_cpuclk_init(struct clk_hw *clk_hw)
{
    struct device_node *np;
    u32 val;

    if((np=of_find_node_by_name(NULL, "cpufreq")))
    {
        if(!of_property_read_u32(np, "vid0-gpio", &val))
            vid_0 = val;
        else
            vid_0 = -1;

        if(!of_property_read_u32(np, "vid1-gpio", &val))
            vid_1 = val;
        else
            vid_1 = -1;
        if(vid_0!=-1 || vid_1!=-1)
            CLK_ERR("[%s] get dvfs gpio %s %s\n", __func__, (vid_0 != -1)?"vid_0":"", (vid_1 != -1)?"vid_1":"");
    }
    else
    {
        vid_0 = -1;
        vid_1 = -1;
        CLK_ERR("[%s] can't get cpufreq node for dvfs\n", __func__);
    }

    // switch vcore control function by bonding id
    ms_vid_ctrl_fn_init();
}


void ms_cpuclk_dvfs_disable(void)
{
    if(vid_0 != -1)
    {
        MDrv_GPIO_Pad_Set(vid_0);
        MDrv_GPIO_Set_High(vid_0);
    }
    if(vid_1 != -1)
    {
        MDrv_GPIO_Pad_Set(vid_1);
        MDrv_GPIO_Set_High(vid_1);
    }
}
EXPORT_SYMBOL(ms_cpuclk_dvfs_disable);

struct clk_ops ms_cpuclk_ops = {
    .round_rate = ms_cpuclk_round_rate,
    .recalc_rate = ms_cpuclk_recalc_rate,
    .set_rate = ms_cpuclk_set_rate,
    .init = ms_cpuclk_init,
};


static void __init ms_clk_complex_init(struct device_node *node)
{
    struct clk *clk;
    struct clk_hw *clk_hw = NULL;
    struct clk_init_data *init = NULL;
    struct clk_ops *clk_ops =NULL;
    const char **parent_names = NULL;
    u32 i;

    clk_hw = kzalloc(sizeof(*clk_hw), GFP_KERNEL);
    init = kzalloc(sizeof(*init), GFP_KERNEL);
    clk_ops = kzalloc(sizeof(*clk_ops), GFP_KERNEL);
    if (!clk_hw || !init || !clk_ops)
        goto fail;

    clk_hw->init = init;
    init->name = node->name;
    init->ops = clk_ops;

    //hook callback ops for cpuclk
    if(!strcmp(node->name, "CLK_cpupll_clk"))
    {
        CLK_ERR("Find %s, hook ms_cpuclk_ops\n", node->name);
        init->ops = &ms_cpuclk_ops;
    }
    else
    {
        CLK_DBG("Find %s, but no ops\n", node->name);
    }

    init->num_parents = of_clk_get_parent_count(node);
    if (init->num_parents < 1)
    {
        CLK_ERR("[%s] %s must have at least one parent\n", __func__, node->name);
        goto fail;
    }

    parent_names = kzalloc(sizeof(char *) * init->num_parents, GFP_KERNEL);
    if (!parent_names)
        goto fail;

    for (i = 0; i < init->num_parents; i++)
        parent_names[i] = of_clk_get_parent_name(node, i);

    init->parent_names = parent_names;
    clk = clk_register(NULL, clk_hw);
    if(IS_ERR(clk))
    {
        CLK_ERR("[%s] Fail to register %s\n", __func__, node->name);
        goto fail;
    }
    else
    {
        CLK_DBG("[%s] %s register successfully\n", __func__, node->name);
    }
    of_clk_add_provider(node, of_clk_src_simple_get, clk);
    clk_register_clkdev(clk, node->name, NULL);
    return;

fail:
    kfree(parent_names);
    kfree(clk_ops);
    kfree(init);
    kfree(clk_hw);
}

CLK_OF_DECLARE(ms_clk_complex, "sstar,complex-clock", ms_clk_complex_init);
